/*	Video for Linux Two
 *	Video Capture Driver
 *	-- example code --
 *
 *	This software is in the public domain.
 *	Written by Bill Dirks
 *
 *	This module is an example implementation of the Video for Linux Two
 *	video capture API specification. The purposes of this software are
 *	1) Serve as a starting point for a new V4L2 capture driver
 *	2) Serve as a dummy driver for an application developer
 *	3) Supplement the V4L2 API documentation 
 *
 *	gcc -c -O2 -Wall v4l2cap.c
 */

#ifndef __KERNEL__
#define __KERNEL__
#endif
#ifndef MODULE
#define MODULE
#endif
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/kernel.h>
#include <linux/malloc.h>
#include <linux/mm.h>
#include <linux/poll.h>
#include <linux/ioport.h>
#include <asm/io.h>
#include <linux/sched.h>
#include <linux/videodev.h>
#include <linux/version.h>
#include <asm/uaccess.h>
#include <linux/pci.h>
#include <asm/pgtable.h>
#include <asm/page.h>
#include <linux/interrupt.h>



#define PKMOD "cap: "
#if 1
#define debug_msg(fmt,arg...) printk(KERN_DEBUG PKMOD fmt,##arg)
#else
#define debug_msg(fmt,arg...)
#endif
#if 1
#define err_msg(fmt,arg...) printk(KERN_ERR PKMOD fmt,##arg)
#else
#define err_msg(fmt,arg...)
#endif
#if 1
#define info_msg(fmt,arg...) printk(KERN_INFO PKMOD fmt,##arg)
#else
#define info_msg(fmt,arg...)
#endif


/*  Video controls  */

static struct v4l2_queryctrl capture_control[] =
{
 {V4L2_CID_BRIGHTNESS, "Brightness", 0, 63, 1, 32, V4L2_CTRL_TYPE_INTEGER},
 {V4L2_CID_CONTRAST,   "Contrast",   0, 63, 1, 32, V4L2_CTRL_TYPE_INTEGER},
 {V4L2_CID_SATURATION, "Saturation", 0, 63, 1, 32, V4L2_CTRL_TYPE_INTEGER},
 /*{V4L2_CID_HUE,        "Hue",        0, 63, 1, 32, V4L2_CTRL_TYPE_INTEGER},*/
#if 1 /* extra stuff used for testing vidpanel */
 {V4L2_CID_EXPOSURE, "Exposure", 0,4,0,0, V4L2_CTRL_TYPE_MENU},
 {V4L2_CID_AUTOGAIN, "Auto Gain", 0, 1, 0, 1, V4L2_CTRL_TYPE_BOOLEAN},
 {V4L2_CID_DO_WHITE_BALANCE, "White Balance", 0,0,0,0, V4L2_CTRL_TYPE_BUTTON},
#endif
};
#define MAXCONTROLS	(sizeof(capture_control)/sizeof(capture_control[0]))
#define VCTRL_BRIGHTNESS	0
#define VCTRL_CONTRAST		1
#define VCTRL_SATURATION	2
#define VCTRL_HUE		3
static int
find_vctrl(int id)
{
        int i;
	if (id == V4L2_CID_PRIVATE_BASE ||
	    id <  V4L2_CID_BASE ||
            id >  V4L2_CID_LASTP1)
		return -EDOM;
        for (i = MAXCONTROLS - 1; i >= 0; i--)
                if (capture_control[i].id == id)
                        break;
	if (i < 0)
		i = -EINVAL;
        return i;
}
static int
vctrl_querymenu(struct v4l2_querymenu	*qm)
{
	static char *expo_menu[] = {
		"1/60",
		"1/100",
		"1/250",
		"1/1000",
		"1/5000",
	};
	if (qm->id == V4L2_CID_EXPOSURE)
	{
		if (qm->index < 0 ||
		    qm->index >= sizeof(expo_menu)/sizeof(char *))
			return -EINVAL;
		{
			memcpy(qm->name, expo_menu[qm->index],
			       sizeof(qm->name));
		}
		return 0;
	}
	return -EINVAL;
}



struct capture_device;/* forward reference */

struct video_decoder
{
	int	is_initialized;
	int	ntsc_hskip;
	int	ntsc_vskip;
	int	ntsc_width;
	int	ntsc_height;
	int	ntsc_field_order;
	int	pal_hskip;
	int	pal_vskip;
	int	pal_width;
	int	pal_height;
	int	pal_field_order;
	int	preferred_field;

	int	num_inputs;
	int	input;
	__u32	standards;
	__u32	standard;
	__u32	frame_period;
	int	decoder_is_stable;
	__u32	decoder_stable_time;

	/*  Changable method functions helps us support multiple	*/
	/*  different types of video decoders easily			*/
	int	(*initialize)(struct capture_device *dev);
	int	(*set_input)(struct capture_device *dev, int x);
	int	(*set_standard)(struct capture_device *dev, int x);
	int	(*set_vcrmode)(struct capture_device *dev, int x);
	int	(*is_stable)(struct capture_device *dev);
};

struct video_source
{
	struct v4l2_input	input;
	int			control[MAXCONTROLS];
	struct v4l2_tuner	tuner;
	int			vcrmode;
};
/*  Indices into the array of video_source's */
#define VSOURCE_COMP		0
#define VSOURCE_SVIDEO		1
//#define VSOURCE_TUNER		3
#define VSOURCE_COUNT		2

/*  Bus-master scatter list  */
struct scatter_node
{
	__u32	addr;
	__u32	len;
};
#define END_OF_SCATTER_LIST	0x80000000

/*  Image translations  */
struct lookup_rgb24
{
	__u32	u_rgb[256];
	__u32	v_rgb[256];
	__u32	y_rgb[256];
	__u8	sat[1024];
};

struct lookup
{
	int	type;
	int	size;
	union
	{
		void			*base;	/* vmalloc() */
		__u16			*rgb16;
		struct lookup_rgb24	*rgb24;
	} table;
};
#define LUT_NULL	0
#define LUT_RGB555	1
#define LUT_RGB565	2
#define LUT_RGB24	3

struct translation
{
	int		type;
	int		width;
	int		height;
	__u8		*in;
	int		in_stride;
	__u8		*out;
	int		out_stride;
	int		output_size;
	int		output_is_user;
	struct lookup	lut;
};
#define XLAT_NULL		0
#define XLAT_YUYV_TO_UYVY	1
#define XLAT_YUYV_TO_YUV420	2
#define XLAT_YUYV_TO_GREY	4
#define XLAT_YUYV_TO_RGB555	6
#define XLAT_YUYV_TO_RGB565	7
#define XLAT_YUYV_TO_RGB24	8
#define XLAT_YUYV_TO_RGB32	9

/*  Per-open data for handling multiple opens on one device */
struct device_open
{
	int			isopen;
	int			noncapturing;
	struct capture_device	*dev;
};
#define MAX_OPENS	3

/*  Streaming data buffer  */
struct stream_buffer
{
	struct v4l2_q_node	qnode;
	struct v4l2_buffer	vidbuf;
	int			requested;
	__u8			*vaddress;  /* vmalloc() */
	struct scatter_node	*dma_list;  /* get_free_page() */
};
#define MAX_CAPTURE_BUFFERS	30
#define MAX_LOCKED_MEMORY	2000000

/*
 *	Capture device structure
 *
 *	One for each handled device in the system.
 *	This structure holds all the global information the driver
 *	needs about each device.
 */
struct capture_device
{
	struct v4l2_device	v;	/*  Must be first */
	char			shortname[16];
	int			is_registered;
	int			open_count;
	struct device_open	open_data[MAX_OPENS];
	int			capturing_opens;

/*	Per-bus index number for each device	*/
	int			index;
/*	General type of device  */
	int			type;
/*	Pointer to the pci_dev structure for this board	*/
	struct pci_dev		*pci;
/*	I/O Base address for non-PCI devices	*/
	unsigned int		iobase;
/*	Interrupts	*/
	int			irq;
	int			ints_enabled;
	struct tq_struct	tqnode_dpc;/* for Bottom Half routine */
	struct timer_list	tlnode;/* for polling interrupts */
	struct wait_queue	*new_video_frame;

/*	Video decoder stuff		*/
	struct video_decoder	videc;
	struct video_source	source[VSOURCE_COUNT];
	int			source_width;
	int			source_height;

/*	Client capture format and capture modes	*/
	struct v4l2_format	clientfmt;
	struct v4l2_captureparm	capture;
	int			input;/* which video source is selected */

/*	Hardware capture format	 */
	int			capture_bypp;
	int			capture_size;
	__u8			*capture_buffer;/* vmalloc() */
	int			capture_buffer_size;
	struct scatter_node	*capture_dma_list;/* get_free_page() */

/*	Capture state	*/
	int			ready_to_capture;
	int			grabber_enabled;
	int			capture_completed;
	unsigned long		time_acquired;/* millisecond time stamp */
	int			streaming;
	struct stream_buffer	stream_buf[MAX_CAPTURE_BUFFERS];
	int			stream_buffers_requested;
	int			stream_buffers_mapped;
	int			stream_contig_map;
	struct v4l2_queue	stream_q_capture;
	struct v4l2_queue	stream_q_done;
	struct timeval		stream_begin;
	unsigned long		stream_last_frame;
	__u8			*stream_capture_buffer;

/*	Image format conversions	*/
	struct translation	translation;
	__u8			*xlat_temp;/* vmalloc() */

/*	Performance statistics	*/
	struct v4l2_performance perf;

	/* frame counter for test images */
	int	h,m,s,f;
};
/*	Values for type field	*/
#define DEVICE_TYPE_0		0
/*	Extreme video dimensions	*/
#define MIN_WIDTH		32
#define MIN_HEIGHT		24
#define MAX_WIDTH		640
#define MAX_HEIGHT		288
#define MAX_FRAME_AGE		200 /* ms */


/*
 *	The Capture device structure array. This is the only global
 *	variable in the module besides those used by the device probing
 *	and enumeration routines (command line overrides)
 */
#define NBOARDS		2
static struct capture_device capture[NBOARDS];
static int unit_video[NBOARDS] = { 0, 1, };
MODULE_PARM(unit_video, "1-"__MODULE_STRING(NBOARDS)"i");

static inline struct capture_device *
capture_device_from_file(struct file *file)
{
	return (struct capture_device *)v4l2_device_from_file(file);
}


/*  These macros can be used to make device I/O operations atomic  */
/* static spinlock_t device_lock = SPIN_LOCK_UNLOCKED; */
/* #define BEGIN_CRITICAL_SECTION	\ */
/*         do{unsigned long flags;spin_lock_irqsave(&wavi_lock,flags) */
/* #define END_CRITICAL_SECTION	\ */
/* 	spin_unlock_irqrestore(&wavi_lock,flags);}while(0) */



/*
 *	D E V I C E   F U N C T I O N S
 */

static void
device_initialize(struct capture_device *dev)
{
	/*  TODO: Put hardware into a sensible state and	*/
	/*        do the one-time startup things		*/
}

static void
device_brightness(struct capture_device *dev, int x)
{
}

static void
device_contrast(struct capture_device *dev, int x)
{
}

static void
device_saturation(struct capture_device *dev, int x)
{
}

static void
device_tone_controls(struct capture_device *dev)
{
	int	*ctrl;
	ctrl = dev->source[dev->input].control;
	device_brightness(dev, ctrl[VCTRL_BRIGHTNESS]);
	device_contrast(dev, ctrl[VCTRL_CONTRAST]);
	device_saturation(dev, ctrl[VCTRL_SATURATION]);
/* 	device_hue(dev, ctrl[VCTRL_HUE]); */
}

static void
grabbing_enable(struct capture_device *dev, int x)
{
	/*  TODO: Start the grabbing hardware  */
}

static int
isqrt(unsigned int q)
{/*	A little integer square root routine */
	int		i;
	int		r;
	unsigned int	b2	= 0x40000000;
	unsigned int	t;

	for (i = 16, r = 0; i > 0 && q; --i)
	{
		t = ((unsigned int)r << i) + b2;
		if (t <= q)
		{
			q -= t;
			r |= (1 << (i - 1));
		}
		b2 >>= 2;
	}
	return r;
}

static unsigned long
current_time_ms(void)
{
	struct timeval	now;

	do_gettimeofday(&now);
	return now.tv_sec * 1000 + now.tv_usec / 1000;
}

#if 0
/*
 *
 *	B U S   M A S T E R   F U N C T I O N S
 *
 */

static int
bm_build_scatter_list(struct capture_device *dev,
		      unsigned char	   *buffer,
		      struct scatter_node **plist)
{
	struct scatter_node	*list;
	int			i, n;
	unsigned char		*a;

	if (buffer == NULL)
		return 0;
	list = *plist;
	if (list == NULL)
	{/*	Assuming one page will be big enough. 4KB = 512 pieces */
		list = (struct scatter_node *)get_free_page(GFP_KERNEL);
		if (list == NULL)
			return 0;
	}

	/*  Simple algorithm will just map the buffer contiguously by pages */
	/*  Note: capture_buffer is vmalloc()ed, so it's page-aligned */

	n = (dev->capture_size + PAGE_SIZE - 1) / PAGE_SIZE;
	a = buffer;
	for (i = 0; i < n; ++i)
	{
		list[i].addr = v4l2_vmalloc_to_bus(a);
		list[i].len = PAGE_SIZE;
		a += PAGE_SIZE;
	}
	/* Last page might not be full */
	if (dev->capture_size < n * PAGE_SIZE)
		list[n - 1].len = dev->capture_size - (n - 1) * PAGE_SIZE;

	list[n - 1].len |= END_OF_SCATTER_LIST;

#if 0
	debug_msg("Scatter list %08lX\n", virt_to_bus(list));
	for (i = 0; i < n; ++i)
		debug_msg("List %2d: A=%08X L=%08X\n",i,
			  list[i].addr,list[i].len);
#endif
	*plist = list;
	return 1;/* ok */
}
#endif


/*
 *
 *	V I D E O   D E C O D E R S
 *
 */

static int
decoder_initialize(struct capture_device *dev)
{
	/*  Video decoder information fields	*/
	dev->videc.standards		= (1 << V4L2_STD_NTSC) |
					  (1 << V4L2_STD_PAL);
	dev->videc.ntsc_hskip		= 30;
	dev->videc.ntsc_vskip		= 12;
	dev->videc.ntsc_width		= 640;
	dev->videc.ntsc_height		= 240;
	dev->videc.ntsc_field_order	= 0;
	dev->videc.pal_hskip		= 62;
	dev->videc.pal_vskip		= 14;
	dev->videc.pal_width		= 768;
	dev->videc.pal_height		= 288;
	dev->videc.pal_field_order	= 0;
	dev->videc.preferred_field	= 0;

	dev->videc.num_inputs		= 2;

	dev->videc.decoder_is_stable	= 1;

	return 1;
}

static int
decoder_set_input(struct capture_device *dev, int i)
{
	dev->input = i;
	//dev->videc.decoder_is_stable = 0;
	/*  TODO: Switch the hardware to the new input  */
	return 1;
}

static int
decoder_set_standard(struct capture_device *dev, int x)
{
	dev->videc.standard = x;
	/*  TODO: Switch the hardware to the new standard  */
	switch (x)
	{
	case V4L2_STD_NTSC:
		dev->videc.frame_period = 333667;
		break;
	case V4L2_STD_PAL:
		dev->videc.frame_period = 400000;
		break;
	case V4L2_STD_SECAM:
		dev->videc.frame_period = 400000;
		break;
	}
	return 1;
}

static int
decoder_set_vcrmode(struct capture_device *dev, int x)
{
	dev->source[dev->input].vcrmode = x;
	/*  TODO: Switch decoder to VCR sync timing mode  */
	return 1;
}

static int
decoder_is_stable(struct capture_device *dev)
{
	/*  TODO: Check if decoder is synced to input  */
	return 1;
}

static int
decoder_probe(struct capture_device *dev)
{
	/*  TODO: Probe I2C bus or whatever for the video decoder */

	/*  Fill in the method fields  */
	dev->videc.initialize = decoder_initialize;
	dev->videc.set_input = decoder_set_input;
	dev->videc.set_standard = decoder_set_standard;
	dev->videc.set_vcrmode = decoder_set_vcrmode;
	dev->videc.is_stable = decoder_is_stable;

	//info_msg("Found decoder chip\n");
	return 1;/*  Found  */
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *
 *	Probe I2C bus for video decoder and fill in the device fields
 */
static int
find_decoder(struct capture_device *dev)
{
	if (!decoder_probe(dev))
		return 0;/*  Failure  */
	return 1;
}

static void
set_video_input(struct capture_device *dev,
		int i)
{
	if (i < 0 || i >= dev->videc.num_inputs)
		return;
	dev->videc.set_input(dev, i);
	dev->videc.set_vcrmode(dev, dev->source[i].vcrmode);
	device_tone_controls(dev);
}


/*
 *
 *	I M A G E   F O R M A T   T R A N S L A T I O N
 *
 */

static int
translate_yuyv_grey(struct translation *xlat)
{
	__u8	*esi, *edi;
	__u32	eax, ebx, ecx, edx;
	int	row;

	esi = xlat->in;
	edi = xlat->out;
	eax = xlat->in_stride - xlat->width * 2;
	ebx = xlat->out_stride - xlat->width;
	for (row = xlat->height; row; esi += eax, edi += ebx, --row)
	{
		for (ecx = xlat->width >> 2; ecx; --ecx)
		{
			edx = esi[4] | (esi[6] << 8);
			edx <<= 16;
			edx |= esi[0] | (esi[2] << 8);
			esi += 8;
			*(__u32 *)edi = edx;
			edi += 4;
		}
	}
	return 1;
}

static int
translate_yuyv_yuv420(struct translation *xlat)
{
	__u8	*esi, *edi;
	__u32	eax, ebx, ecx, edx;
	__u8	dl;
	int	row;

	/* Y's */
	esi = xlat->in;
	edi = xlat->out;
	eax = xlat->in_stride - xlat->width * 2;
	ebx = xlat->out_stride - xlat->width;
	for (row = xlat->height; row; esi += eax, edi += ebx, --row)
	{
		for (ecx = xlat->width >> 2; ecx; --ecx)
		{
			edx = esi[4] | (esi[6] << 8);
			edx <<= 16;
			edx |= esi[0] | (esi[2] << 8);
			esi += 8;
			*(__u32 *)edi = edx;
			edi += 4;
		}
	}
	/* U's */
	esi = xlat->in + xlat->in_stride;
	eax = xlat->in_stride * 2 - xlat->width * 2;
	ebx >>= 1;
	for (row = xlat->height >> 1; row; esi += eax, edi += ebx, --row)
	{
		for (ecx = xlat->width >> 1; ecx; ++edi, --ecx)
		{
			dl = esi[1];
			esi += 4;
			*edi = dl;
		}
	}
	/* V's */
	esi = xlat->in + xlat->in_stride;
	for (row = xlat->height >> 1; row; esi += eax, edi += ebx, --row)
	{
		for (ecx = xlat->width >> 1; ecx; ++edi, --ecx)
		{
			dl = esi[3];
			esi += 4;
			*edi = dl;
		}
	}
	return 1;
}

#define K12_1	 4096
#define K12_S	   12
#define K12_GU	-1409
#define K12_BU	 7258
#define K12_RV	 5743
#define K12_GV	-2925
static int
translate_expand_y(int y)
{
	y = (255 * y + 110) / 220;
	if (y < 0) y = 0; else if (y > 255) y = 255;
	return y;
}
static int
translate_expand_c(int c)
{
	c = (127 * c + 56) / 112;
	if (c < -128) c = -128; else if (c > 127) c = 127;
	return c;
}

static int
translate_make_rgb16_lut(struct translation *xlat)
{
	__u16	*lut;
	long	gu[32], bu[32], rv[32], gv[32];
	int	rscale[256], gscale[256], bscale[256];
	int	rrange, grange, brange;
	int	rshift, gshift, bshift;
	long	x;
	int	y, u, v;
	int	r, g, b;
	int	i, t;

	if ((xlat->type == XLAT_YUYV_TO_RGB555 &&
	     xlat->lut.type == LUT_RGB555) ||
	    (xlat->type == XLAT_YUYV_TO_RGB565 &&
	     xlat->lut.type == LUT_RGB565))
		return 1;

	if (xlat->lut.table.base)
		vfree(xlat->lut.table.base);
	xlat->lut.table.base = vmalloc(1 << 17);
	if (xlat->lut.table.base == NULL)
	{
		err_msg("vmalloc() failed in make_rgb16_lut\n");
		return 0;
	}
	lut = xlat->lut.table.rgb16;

	//  Compute all different chroma components to 8-bit precision
	for (i = 0, t = -128; t < 128; t += 8, ++i)
	{
		x = translate_expand_c(t) + 2;
		gu[i] = (K12_GU * x + K12_1/2) >> K12_S;
		bu[i] = (K12_BU * x + K12_1/2) >> K12_S;
		rv[i] = (K12_RV * x + K12_1/2) >> K12_S;
		gv[i] = (K12_GV * x + K12_1/2) >> K12_S;
	}
	//  8-bit to ?-bit scaling tables
	if (xlat->type == XLAT_YUYV_TO_RGB555)
	{
		xlat->lut.type = LUT_RGB555;
		rrange = grange = brange = 31;
		rshift = 10; gshift = 5; bshift = 0;
	}
	else
	{
		xlat->lut.type = LUT_RGB565;
		rrange = brange = 31;
		grange = 63;
		rshift = 11; gshift = 5; bshift = 0;
	}
	for (i = 0; i < 256; ++i)
	{
		rscale[i] = ((i * rrange + 127) / 255) << rshift;
		gscale[i] = ((i * grange + 127) / 255) << gshift;
		bscale[i] = ((i * brange + 127) / 255) << bshift;
	}

	//  Fill in the RGB values for each combination of YUV
	for (i = 0; i < 256; i += 4)
	{
		y = translate_expand_y(i) + 2;
		if (y > 255) y = 255;
		for (u = 0; u < 32; ++u)
			for (v = 0; v < 32; ++v)
			{
				//  Red, Green and Blue
				r = y + rv[v];
				g = y + gu[u] + gv[v];
				b = y + bu[u];
				//  Saturate
				if (r < 0) r = 0; else if (r > 254) r = 254;
				if (g < 0) g = 0; else if (g > 254) g = 254;
				if (b < 0) b = 0; else if (b > 254) b = 254;
				//  scale, shift and combine
				*lut++ = rscale[r] + gscale[g] + bscale[b];
			}
	}
	return 1;
}

static int
translate_yuyv_rgb16(struct translation *xlat)
{
	__u32	*src, *dst;
	__u32	uv, yuv0, yuv1, dual;
	__u16	*lut;
	int	stride;
	int	row, i;

	if (!translate_make_rgb16_lut(xlat))
		return 0;
	lut = xlat->lut.table.rgb16;
	src = (__u32 *)xlat->in;
	dst = (__u32 *)xlat->out;
	if (src == NULL || dst == NULL)
		return 0;
	stride = (xlat->out_stride - xlat->width * 2) >> 2;
	for (row = xlat->height; row; --row)
	{
		for (i = xlat->width >> 1; i; --i)
		{
			dual = *src++;
			uv   =    ((dual & 0x0000F800) >> 6)
				+ ((dual & 0xF8000000) >> 27);
			yuv1 = 	  ((dual & 0x00FC0000) >> 8) + uv;
			yuv0 = 	  ((dual & 0x000000FC) << 8) + uv;
			*dst++ = ((__u32)lut[yuv1] << 16) | lut[yuv0];
		}
		dst += stride;
	}
	return 1;
}

static int
translate_make_rgb24_lut(struct translation *xlat)
{
	struct lookup_rgb24	*lut;
	int			r, g, b;
	int			i;
	int			x;

	if (xlat->lut.type == LUT_RGB24)
		return 1;

	if (xlat->lut.table.base)
		vfree(xlat->lut.table.base);
	xlat->lut.table.base = vmalloc(sizeof(struct lookup_rgb24));
	if (xlat->lut.table.base == NULL)
	{
		err_msg("vmalloc() failed in make_rgb24_lut\n");
		return 0;
	}
	xlat->lut.type = LUT_RGB24;
	lut = xlat->lut.table.rgb24;

	for (i = 0; i < 256; ++i)
	{
		x = i;		// Value is in excess-128 format
		if (x < 128)
			++x;	// Add 1 to negative values for noise rejection
		x -= 128;	// Convert to two's complement format
		x = translate_expand_c(x);

		g = (K12_GU * x + K12_1/2) >> K12_S;
		b = (K12_BU * x + K12_1/2) >> K12_S;
		lut->u_rgb[i] = ((g & 0x3FF) << 11) | (b & 0x3FF);

		r = (K12_RV * x + K12_1/2) >> K12_S;
		g = (K12_GV * x + K12_1/2) >> K12_S;
		lut->v_rgb[i] = (r << 22) | ((g & 0x3FF) << 11);
                
		x = translate_expand_y(i);
		lut->y_rgb[i] = (x << 22) | (x << 11) | x;
	}
	for (i = 0; i < 1024; ++i)
	{
		x = (i > 511) ? 0 : ((i > 255) ? 255 : i);
		lut->sat[i] = x;
	}
	return 1;
}

static int
translate_yuyv_rgb24(struct translation *xlat)
{
	struct lookup_rgb24	*lut;
	__u8			*src;
	__u32			*dst;
	int			i;
	int			row;
	int			stride;
	__u32			pela, pelb, pelc, peld;

	if (!translate_make_rgb24_lut(xlat))
		return 0;
//debug_msg("converting to rgb24\n");
	lut = xlat->lut.table.rgb24;
	src = (__u8 *)xlat->in;
	dst = (__u32 *)xlat->out;
	if (src == NULL || dst == NULL)
		return 0;
	stride = (xlat->out_stride - 3 * xlat->width) >> 2;
	for (row = xlat->height; row; --row)
	{
		for (i = xlat->width >> 2; i; --i)
		{
			pelb  = lut->u_rgb[src[1]]
			      + lut->v_rgb[src[3]];
			pela  = lut->y_rgb[src[0]] + pelb;
			pelb += lut->y_rgb[src[2]];
			peld  = lut->u_rgb[src[5]]
			      + lut->v_rgb[src[7]];
			pelc  = lut->y_rgb[src[4]] + peld;
			peld += lut->y_rgb[src[6]];
			src += 8;

			dst[0] = ((u32)lut->sat[pela & 0x3FF])
			       + ((u32)lut->sat[(pela >> 11) & 0x3FF] << 8)
			       + ((u32)lut->sat[pela >> 22] << 16)
			       + ((u32)lut->sat[pelb & 0x3FF] << 24);
			dst[1] = ((u32)lut->sat[(pelb >> 11) & 0x3FF])
			       + ((u32)lut->sat[pelb >> 22] << 8)
			       + ((u32)lut->sat[pelc & 0x3FF] << 16)
			       + ((u32)lut->sat[(pelc >> 11) & 0x3FF] << 24);
			dst[2] = ((u32)lut->sat[pelc >> 22])
			       + ((u32)lut->sat[peld & 0x3FF] << 8)
			       + ((u32)lut->sat[(peld >> 11) & 0x3FF] << 16)
			       + ((u32)lut->sat[peld >> 22] << 24);
			dst += 3;
		}
		dst += stride;
	}

	return 1;
}

static int
translate_yuyv_rgb32(struct translation *xlat)
{
	struct lookup_rgb24	*lut;
	__u8			*src;
	__u32			*dst;
	int			i;
	int			row;
	int			stride;
	__u32			pela, pelb;

	if (!translate_make_rgb24_lut(xlat))
		return 0;
//debug_msg("converting to rgb32\n");
	lut = xlat->lut.table.rgb24;
	src = (__u8 *)xlat->in;
	dst = (__u32 *)xlat->out;
	if (src == NULL || dst == NULL)
		return 0;
	stride = (xlat->out_stride - 4 * xlat->width) >> 2;
	for (row = xlat->height; row; --row)
	{
		for (i = xlat->width >> 1; i; --i)
		{
			pelb  = lut->u_rgb[src[1]]
			      + lut->v_rgb[src[3]];
			pela  = lut->y_rgb[src[0]] + pelb;
			pelb += lut->y_rgb[src[2]];
			src += 4;
			
			dst[0] =  lut->sat[pela & 0x3FF]
			       + (lut->sat[(pela >> 11) & 0x3FF] << 8)
			       + (lut->sat[pela >> 22] << 16);
			dst[1] =  lut->sat[pelb & 0x3FF]
			       + (lut->sat[(pelb >> 11) & 0x3FF] << 8)
			       + (lut->sat[pelb >> 22] << 16);
			dst += 2;
		}
		dst += stride;
	}

	return 1;
}


static void
translate_close(struct capture_device *dev)
{
	dev->translation.type = XLAT_NULL;
	dev->translation.in = NULL;
	dev->translation.out = NULL;
	dev->translation.lut.type = LUT_NULL;
	if (dev->translation.lut.table.base)
		vfree(dev->translation.lut.table.base);
	dev->translation.lut.table.base = NULL;

	if (dev->xlat_temp)
		vfree(dev->xlat_temp);
	dev->xlat_temp = NULL;
}

static int
translate_setup(struct capture_device *dev)
{
	int	npix2;
	translate_close(dev);

	/*  Translation: YUYV to client format */
	dev->translation.width = dev->clientfmt.width;
	dev->translation.height = dev->clientfmt.height;
	dev->translation.in_stride = dev->translation.width * 2;
	dev->translation.output_size = dev->clientfmt.sizeimage;

	npix2 = dev->translation.width * dev->translation.height;
	switch (dev->clientfmt.pixelformat)
	{
	case V4L2_PIX_FMT_YUYV:
		dev->translation.type = XLAT_NULL;
		break;
	case V4L2_PIX_FMT_GREY:
		dev->translation.type = XLAT_YUYV_TO_GREY;
		dev->translation.out_stride = dev->translation.width;
		break;
	case V4L2_PIX_FMT_YUV420:
		dev->translation.type = XLAT_YUYV_TO_YUV420;
		dev->translation.out_stride = dev->translation.width;
		break;
	case V4L2_PIX_FMT_RGB555:
	case V4L2_PIX_FMT_RGB565:
		dev->translation.type = (dev->clientfmt.pixelformat == 
			V4L2_PIX_FMT_RGB555) ? XLAT_YUYV_TO_RGB555 
			: XLAT_YUYV_TO_RGB565;
		dev->translation.out_stride = dev->translation.width * 2;
		if (!translate_make_rgb16_lut(&dev->translation))
			return 0;
		break;
	case V4L2_PIX_FMT_BGR24:
		dev->translation.type = XLAT_YUYV_TO_RGB24;
		dev->translation.out_stride = dev->translation.width * 3;
		if (!translate_make_rgb24_lut(&dev->translation))
			return 0;
		break;
	case V4L2_PIX_FMT_BGR32:
		dev->translation.type = XLAT_YUYV_TO_RGB32;
		dev->translation.out_stride = dev->translation.width * 4;
		if (!translate_make_rgb24_lut(&dev->translation))
			return 0;
		break;
	}
	return 1;
}

static void
translate_inandout(struct capture_device *dev,
		   __u8 *input_buffer,
		   __u8 *output_buffer,
		   __u8 output_is_user_space)
{
	/*  Translation: YUYV to client format */
	dev->translation.in = input_buffer;
	dev->translation.out = output_buffer;
	dev->translation.output_is_user = output_is_user_space;
}

static int /* length of output image or negative error */
translate_image(struct capture_device *dev,
		__u8	*input_buffer,
		__u8	*output_buffer,
		int	len,
		int	output_is_user)
{
	int	err;

	/* The buffer must be large enough for the whole image */
	if (len < dev->translation.output_size)
	{
		debug_msg("Read buffer too small, %d < %d\n",
			  len, dev->translation.output_size);
		return -EFAULT;
	}
	if (len > dev->translation.output_size)
		len = dev->translation.output_size;

	translate_inandout(dev, input_buffer, output_buffer, output_is_user);

	/*  Translation: YUYV to client format */
	if (dev->translation.type == XLAT_NULL)
	{
		if (dev->translation.in == dev->translation.out)
			return len;
		if (!output_is_user)
			memcpy(output_buffer, dev->translation.in, len);
		else
		{
			err = copy_to_user(output_buffer, 
					   dev->translation.in, len);
			len = (err) ? -EFAULT : len;
		}
		return len;
	}

	if (output_is_user && !access_ok(VERIFY_WRITE, output_buffer, len))
	{
		debug_msg("Buffer verify failed in translate_image\n");
		return -EFAULT;
	}

	switch (dev->translation.type)
	{
	case XLAT_YUYV_TO_GREY:
		translate_yuyv_grey(&dev->translation);
		break;
	case XLAT_YUYV_TO_YUV420:
		translate_yuyv_yuv420(&dev->translation);
		break;
	case XLAT_YUYV_TO_RGB555:
	case XLAT_YUYV_TO_RGB565:
		translate_yuyv_rgb16(&dev->translation);
		break;
	case XLAT_YUYV_TO_RGB24:
		translate_yuyv_rgb24(&dev->translation);
		break;
	case XLAT_YUYV_TO_RGB32:
		translate_yuyv_rgb32(&dev->translation);
		break;
	default:
		debug_msg("Unknown image translation\n");
		break;
	}
	dev->translation.out = NULL;
	return len;
}

/*
 *
 *	V I D E O   C A P T U R E   F U N C T I O N S
 *
 */

/*
 *  Supported capture formats
 */
static struct v4l2_fmtdesc capfmt[] = 
{
	{	0, {"RGB-16 (5-5-5)"},
		V4L2_PIX_FMT_RGB555,  0, 16, {0, 0},
	},
	{	1, {"RGB-16 (5-6-5)"},
		V4L2_PIX_FMT_RGB565,  0, 16, {0, 0},
	},
	{	2, {"RGB-24 (B-G-R)"},
		V4L2_PIX_FMT_BGR24,   0, 24, {0, 0},
	},
	{	3, {"RGB-32 (B-G-R-?)"},
		V4L2_PIX_FMT_BGR32,   0, 32, {0, 0},
	},
	{	4, {"Greyscale-8"},
		V4L2_PIX_FMT_GREY,    V4L2_FMT_CS_601YUV, 8, {0, 0},
	},
	{	5, {"YUV 4:2:2 (Y-U-Y-V)"},
		V4L2_PIX_FMT_YUYV,    V4L2_FMT_CS_601YUV, 16, {0, 0},
	},
	{	6, {"YUV 4:2:0 (planar)"},
		V4L2_PIX_FMT_YUV420,  V4L2_FMT_CS_601YUV, 12, {0, 0},
	},
};
#define NUM_CAPFMT (sizeof(capfmt)/sizeof(capfmt[0]))


static void interrupt_enable(struct capture_device *dev);

/*  The image format has changed, width, height, pixel format.
 *  Decide if the format is ok or take the closest valid format.
 */
static int
capture_new_format(struct capture_device *dev)
{
	int	ntsc;
	int	max_image_size;
	int	max_height;
	int	max_width;
	int	max_pixels;
	int	t;

	ntsc = (dev->videc.standard == V4L2_STD_NTSC);
	dev->ready_to_capture = 0;

	dev->source_width = 
		(ntsc) ? dev->videc.ntsc_width : dev->videc.pal_width;
	dev->source_height = 
		(ntsc) ? dev->videc.ntsc_height : dev->videc.pal_height;

	dev->clientfmt.flags = V4L2_FMT_CS_601YUV;
	switch (dev->clientfmt.pixelformat)
	{
	case V4L2_PIX_FMT_GREY:
		dev->clientfmt.depth = 8;
		break;
	case V4L2_PIX_FMT_YUV420:
		dev->clientfmt.depth = 12;
		break;
	case V4L2_PIX_FMT_RGB555:
	case V4L2_PIX_FMT_RGB565:
		dev->clientfmt.flags = 0;
		/* fall thru */
	case V4L2_PIX_FMT_YUYV:
	case V4L2_PIX_FMT_UYVY:
		dev->clientfmt.depth = 16;
		break;
	case V4L2_PIX_FMT_BGR24:
		dev->clientfmt.depth = 24;
		dev->clientfmt.flags = 0;
		break;
	case V4L2_PIX_FMT_BGR32:
		dev->clientfmt.depth = 32;
		dev->clientfmt.flags = 0;
		break;
	default:
		debug_msg("unknown format %4.4s\n",
			  (char *)&dev->clientfmt.pixelformat);
		dev->clientfmt.depth = 16;
		dev->clientfmt.pixelformat = V4L2_PIX_FMT_YUYV;
		dev->clientfmt.flags = 0;
		break;
	}

	dev->capture_bypp = 2;

	if (dev->clientfmt.width < MIN_WIDTH)
		dev->clientfmt.width = MIN_WIDTH;
	if (dev->clientfmt.height < MIN_HEIGHT)
		dev->clientfmt.height = MIN_HEIGHT;

#define MAX_IMAGE_SIZE 850000
	max_image_size = MAX_IMAGE_SIZE;
	if (dev->stream_buffers_mapped)
	{/*	Limited by size of existing buffers  */
		max_image_size = dev->stream_buf[0].vidbuf.length;
	}
	max_width = dev->source_width;
	max_height = dev->source_height;
	max_pixels = max_image_size / dev->capture_bypp;
	t = isqrt((max_pixels * dev->clientfmt.width) / dev->clientfmt.height);
	if (t < max_width)
		max_width = t;
	t = isqrt((max_pixels * dev->clientfmt.height) / dev->clientfmt.width);
	if (t < max_height)
		max_height = t;

	if (dev->clientfmt.width > max_width)
		dev->clientfmt.width = max_width;
	if (dev->clientfmt.height > max_height)
		dev->clientfmt.height = max_height;

	dev->clientfmt.width &= ~3;
	dev->clientfmt.height &= ~3;
	dev->clientfmt.sizeimage = (dev->clientfmt.width
		* dev->clientfmt.height
		* dev->clientfmt.depth)
		/ 8;

	dev->capture_size = dev->clientfmt.width
		* dev->clientfmt.height
		* dev->capture_bypp;

	/*  TODO: Any other driver state related to the image format  */
	return 1;
}

/*  Stop the music!
 */
static void
capture_abort(struct capture_device *dev)
{
	dev->grabber_enabled = 0;
	/*  Turn off the capture hardware  */
	grabbing_enable(dev, 0);
}

/*  Allocate buffers, and get everything ready to capture
 *  an image, but don't start capturing yet.
 */
static int
capture_begin(struct capture_device *dev)
{
	capture_abort(dev);
	if (dev->ready_to_capture)
		return dev->ready_to_capture;

	if (dev->capture_buffer_size < dev->capture_size)
	{
		if (dev->capture_buffer != NULL)
			vfree(dev->capture_buffer);
		dev->capture_buffer_size = 
			(dev->capture_size + PAGE_SIZE - 1) & ~(PAGE_SIZE - 1);
		dev->capture_buffer = (__u8 *)
			vmalloc(dev->capture_buffer_size);
		if (dev->capture_buffer == NULL)
		{
			dev->capture_buffer_size = 0;
			err_msg("Can't allocate capture buffer"
				" %d bytes\n", dev->capture_size);
			return dev->ready_to_capture;
		}
	}

#if 0
	if (dev->irq && (can use DMA))
		bm_build_scatter_list(dev, dev->capture_buffer, 
				      &dev->capture_dma_list);
#endif

	if (!translate_setup(dev))
		return dev->ready_to_capture;

	interrupt_enable(dev);
debug_msg("Ready to capture!\n");
	return (dev->ready_to_capture = 1);
}

/*  Start an image capture
 */
static void
capture_grab_frame(struct capture_device *dev)
{
	if (dev->ready_to_capture && dev->grabber_enabled)
		return;

	capture_begin(dev);
	if (!dev->ready_to_capture)
		return;

	/*  TODO: Prepare the hardware for the next capture  */

	//list = dev->capture_dma_list;/* DMA list for capture_buffer */

	/*  Set up stream_capture_buffer to point to the buffer to  */
	/*  capture the next frame into  */
	if (dev->streaming)
	{
		struct stream_buffer	*buf;

		dev->stream_capture_buffer = dev->capture_buffer;

		/* Go straight into streaming buffer? */
		if (dev->translation.type == XLAT_NULL)
		{
			buf = v4l2_q_peek_head(&dev->stream_q_capture);
			if (buf != NULL)
			{
				dev->stream_capture_buffer = buf->vaddress;
				//list = buf->dma_list;
			}
		}
	}

	/*  Start the hardware  */
	grabbing_enable(dev, 1);

	dev->grabber_enabled = 1;
	dev->capture_completed = 0;
}

/*
 *	STREAMING CAPTURE
 */

static int/* 1 = success; 0 = failed */
capture_queuebuffer(struct capture_device *dev,
		    struct v4l2_buffer	  *vidbuf)
{
	int			i	= vidbuf->index;
	struct stream_buffer	*buf	= NULL;

	if (!dev->stream_buffers_mapped)
	{
		debug_msg("QBUF no buffers mapped\n");
		return 0;
	}
	if (vidbuf->type != V4L2_BUF_TYPE_CAPTURE)
	{
		debug_msg("QBUF wrong type\n");
		return 0;
	}
	if (i < 0 || i >= MAX_CAPTURE_BUFFERS || !dev->stream_buf[i].requested)
	{
		debug_msg("QBUF buffer index %d is out of range\n", i);
		return 0;
	}

	buf = &dev->stream_buf[i];

	if (!(buf->vidbuf.flags & V4L2_BUF_FLAG_MAPPED))
	{
		debug_msg("QBUF buffer %d is not mapped\n", i);
		return 0;
	}
	if ((buf->vidbuf.flags & V4L2_BUF_FLAG_QUEUED))
	{
		debug_msg("QBUF buffer %d is already queued\n", i);
		return 0;
	}

	buf->vidbuf.flags &= ~V4L2_BUF_FLAG_DONE;
	v4l2_q_add_tail(&dev->stream_q_capture, &buf->qnode);
	buf->vidbuf.flags |= V4L2_BUF_FLAG_QUEUED;
	return 1;
}

static int/* 1 = got a buffer; 0 = no buffers */
capture_dequeuebuffer(struct capture_device *dev,
		      struct v4l2_buffer *buf)
{
	struct stream_buffer *newbuf;

	if (buf->type != V4L2_BUF_TYPE_CAPTURE)
	{
		debug_msg("DQBUF wrong buffer type\n");
		return 0;
	}
	newbuf = v4l2_q_del_head(&dev->stream_q_done);
	if (newbuf == NULL)
	{
		debug_msg("DQBUF nothing on done queue\n");
		return 0;
	}
	newbuf->vidbuf.flags &= ~V4L2_BUF_FLAG_QUEUED;

	*buf = newbuf->vidbuf;
	return 1;
}

static int
capture_streamon(struct capture_device	*dev,
		 __u32			type)
{
	struct stream_buffer *buf;

	if (type != V4L2_BUF_TYPE_CAPTURE)
	{
		debug_msg("STREAMON wrong buffer type\n");
		return 0;
	}
	if (dev->streaming)
	{
		capture_grab_frame(dev);
		return 1;
	}

	capture_abort(dev);/* cancel any capture that might be in progress */

	/*  -2 is a magic number that triggers start-of-stream logic in */
	/*    capture_interrupt()  */
	dev->stream_last_frame = -2;

	dev->perf.frames = 0;
	dev->perf.framesdropped = 0;
	dev->perf.bytesout = 0;

	/*  Can't capture frames faster than the video input  */
	if (dev->capture.timeperframe < dev->videc.frame_period)
		dev->capture.timeperframe = dev->videc.frame_period;

	/*  Move any leftover DONE buffers to the free pool */
	while ((buf = v4l2_q_del_head(&dev->stream_q_done)))
		buf->vidbuf.flags &= ~V4L2_BUF_FLAG_QUEUED;

	/*  Kick off the machine */
	dev->streaming = 1;
	capture_grab_frame(dev);
	return 1;
}

static void
capture_streamoff(struct capture_device	*dev,
		 __u32			type)
{
	if (!dev->streaming)
		return;
	if (type != V4L2_BUF_TYPE_CAPTURE)
	{
		debug_msg("STREAMOFF wrong buffer type\n");
		return;
	}
	capture_abort(dev);
	dev->streaming = 0;

	/* Note: should really delay this till next capture */
	dev->perf.frames = 0;
	dev->perf.framesdropped = 0;
	dev->perf.bytesout = 0;
}


static void
synthetic_image(struct capture_device *dev, __u8 *capture_buffer);

/*	Read out and convert the next frame
 */
static int /* returns length of data or negative for error */
capture_imagereadout(struct capture_device	*dev,
		     __u8			*capture_buffer,
		     __u8			*output_buffer,
		     int			output_size,
		     int			output_is_user)
{
	int	len;

	len = dev->capture_size;
	/*  If not using DMA then get data into capture buffer  */
	{
		/*  In this example driver we have no hardware to get  */
		/*  data from, so we will make a synthetic image  */
		synthetic_image(dev, capture_buffer);
	}
	/*  else if using DMA the image is already in the buffer  */
	/*  Set 'len' to 0 or negative if frame is no good  */

	dev->grabber_enabled = 0;
	grabbing_enable(dev, 0);

	if (len <= 0)
	{
		return len;
	}

	len = translate_image(dev, capture_buffer, output_buffer,
			      output_size, output_is_user);
	if (len < 0)
		return len;
	++dev->perf.frames;
	dev->perf.bytesout += len;
	return len;
}

/*  The hardware has issued the interrupt signal, do any post-capture
 *  processing that may be necessary.
 *  [This function is called indirectly through the immediate task queue;
 *  it executes at elevated IRQL, but it is interruptible. (It's a b.h.)]
 */
static void
capture_interrupt(void *v)
{
	struct capture_device	*dev = (struct capture_device *)v;
	struct stream_buffer	*buf;
	int			len;
	struct timeval		timestamp_rough;
	unsigned long		raw_frame_num;
	unsigned long		next_raw_frame_to_keep;
	unsigned long		stream_frame_num;
	u64			temp64;

	/*  TODO: Check for an interrupt pending on the device, and  */
	/*        return if there is no interrupt pending  */
	/*  (In this hardware-less demo I'll just check the completed flag) */
	if (!dev->grabber_enabled ||
	    dev->capture_completed)
		return;

	if (dev->capture_buffer == NULL || !dev->ints_enabled)
	{
		err_msg("Can't process the interrupt\n");
		return;
	}
	dev->capture_completed = 1;

	if (!dev->streaming)
	{
		dev->time_acquired = current_time_ms();
		/* DMA might not have finished, but we'll check in read() */
		//debug_msg("New frame ready\n");
		wake_up_interruptible(&dev->new_video_frame);
		return;
	}

	/*  Only get here in streaming mode  */

	if (dev->stream_last_frame == -2)
	{/*	First frame of the stream  */
		v4l2_masterclock_gettime(&dev->stream_begin);
		dev->stream_last_frame = -1;
	}

	if (dev->stream_capture_buffer == NULL)
	{
		return;
	}

	buf = v4l2_q_peek_head(&dev->stream_q_capture);
	if (buf == NULL)
	{/*	No available buffers. Skip this frame. This is not an  */
	 /*	error, it's a normal way to throttle the capture rate  */
		dev->grabber_enabled = 0;
		grabbing_enable(dev, 0);
		capture_grab_frame(dev);
		return;
	}

	/*  Compute current stream time  */
	v4l2_masterclock_gettime(&timestamp_rough);
	v4l2_timeval_delta(&timestamp_rough,
			   &dev->stream_begin, &timestamp_rough);

	/*  Capture rate control  */
	if (dev->videc.frame_period == 0)
	{
		err_msg("videc.frame_period == 0 -- divide by 0\n");
		dev->videc.frame_period = 333667;
	}
	raw_frame_num = v4l2_timeval_divide(
		&timestamp_rough, dev->videc.frame_period);
	temp64 = (u64)dev->capture.timeperframe
		* (dev->stream_last_frame + 1)
		+ (dev->videc.frame_period >> 1);
	next_raw_frame_to_keep = 
		v4l2_math_div6432(temp64, dev->videc.frame_period, NULL);
	if (raw_frame_num < next_raw_frame_to_keep)
	{/*	Not time yet, don't keep this frame */
		dev->grabber_enabled = 0;
		grabbing_enable(dev, 0);
		capture_grab_frame(dev);
		return;
	}

	/*  Want this frame  */
	len = capture_imagereadout(dev, dev->stream_capture_buffer,
				   buf->vaddress, buf->vidbuf.length, 0);

	if (len <= 0)
	{/*	Frame no good, DMA did not finish, etc. */
		/*  Begin capturing the next frame now  */
		capture_grab_frame(dev);
		return;
	}

	/*  Fill in the buffer information fields  */
	buf->vidbuf.bytesused = len;
	buf->vidbuf.flags |= V4L2_BUF_FLAG_DONE | V4L2_BUF_FLAG_KEYFRAME;
	buf->vidbuf.timestamp = timestamp_rough;
	stream_frame_num = v4l2_timeval_correct(&buf->vidbuf.timestamp,
						dev->capture.timeperframe);
	//debug_msg("Stream frame %4lu T= %lu.%06lu\n", stream_frame_num,
	//	  buf->vidbuf.timestamp.tv_sec,buf->vidbuf.timestamp.tv_usec);
	if (stream_frame_num > dev->stream_last_frame + 1)
	{/*	We have missed one or more frames  */
		dev->perf.framesdropped += stream_frame_num
			- (dev->stream_last_frame + 1);
	}
	dev->stream_last_frame = stream_frame_num;

	/*  Move buffer to done queue  */
	buf = v4l2_q_del_head(&dev->stream_q_capture);
	v4l2_q_add_tail(&dev->stream_q_done, &buf->qnode);

	/*  Begin capturing the next frame now  */
	capture_grab_frame(dev);
	/*  A new frame is ready!  */
	wake_up_interruptible(&dev->new_video_frame);
}


/*  Read captured data into a user buffer.
 *  Return: negative = error
 *	    0        = keep waiting
 *	    positive = count of bytes read successfully
 */
static long
capture_read(struct capture_device *dev,
	     __u8	*user_buffer,
	     int	user_buffer_size)
{
	int		len = user_buffer_size;
	unsigned long	now;

	if (!dev->ints_enabled)
		return -EIO;

	if (!dev->capture_completed)
	{/* No interrupt has occurred yet, or DMA didn't finish.  */
		//debug_msg("No data ready.\n");
		if (!dev->grabber_enabled)
			capture_grab_frame(dev);
		return 0;/* caller should keep waiting */
	}
	
	now = current_time_ms();
	if (now - dev->time_acquired > MAX_FRAME_AGE)
	{/* Frame in buffer is stale, get a new one */
		debug_msg("Stale frame, re-acquiring.\n");
		dev->grabber_enabled = 0;
		grabbing_enable(dev, 0);
		capture_grab_frame(dev);
		return 0;/* caller should keep waiting */
	}

	len = capture_imagereadout(dev, dev->capture_buffer,
				   user_buffer, user_buffer_size, 1);
	capture_grab_frame(dev);
	return len;
}

/*  Stop capturing and free all resources used for capture.
 */
static void
capture_close(struct capture_device *dev)
{
  //	int	i;

	if (dev->streaming)
		capture_streamoff(dev, V4L2_BUF_TYPE_CAPTURE);
	capture_abort(dev);
	dev->ready_to_capture = 0;
	translate_close(dev);
	if (dev->capture_dma_list)
		free_page((unsigned long)dev->capture_dma_list);
	dev->capture_dma_list = 0;
	if (dev->capture_buffer != NULL)
		vfree(dev->capture_buffer);
	dev->capture_buffer = NULL;
	dev->capture_buffer_size = 0;
#if 0
	for (i = 0; i < MAX_CAPTURE_BUFFERS; ++i)
	{
		dev->stream_buf[i].requested = 0;
		if (dev->stream_buf[i].vaddress)
			vfree(dev->stream_buf[i].vaddress);
		dev->stream_buf[i].vaddress = NULL;
		if (dev->stream_buf[i].dma_list)
			free_page((unsigned long)dev->stream_buf[i].dma_list);
		dev->stream_buf[i].dma_list = NULL;
	}
#endif
}

/*
 *
 *	I N T E R R U P T   R O U T I N E S
 *
 */

/*  This function runs at interrupt time, either in response to a hardware
 *  interrupt, or on each timer tick if there is no hardware interrupt.
 */
static void
interrupt_handler(void *v)
{
	struct capture_device *dev = (struct capture_device *)v;

	if (!dev->ints_enabled)
		return;

	/*  Call "bottom half" of handler  */
	dev->tqnode_dpc.routine = capture_interrupt;
	dev->tqnode_dpc.data = dev;
	queue_task(&dev->tqnode_dpc, &tq_immediate);
	mark_bh(IMMEDIATE_BH);

	if (dev->tlnode.function != NULL &&
	    dev->ints_enabled)
	{/*	Poll again on next timer tick  */
		dev->tlnode.expires = jiffies + HZ/100;
		add_timer(&dev->tlnode);
	}
}

/*  Called by the system when our hardware interrupt occurs
 */
static void
interrupt_hw(int irq, void *v,
	     struct pt_regs *regs)
{
/*	struct capture_device *dev = (struct capture_device *)v; */

	if (v == NULL)
		return;

/* 	if (!(our device has interrupt pending)) */
/* 	{ */
/* 		Not us. Likely another device is sharing the IRQ */
/* 		debug_msg("HW interrupt (not this device)\n"); */
/* 		return; */
/* 	} */

	//debug_msg("HW interrupt (real)\n");
	interrupt_handler(v);
}

static void
interrupt_disable(struct capture_device *dev)
{
	if (!dev->ints_enabled)
		return;
	dev->ints_enabled = 0;

	/*  TODO: Disable interrupts on the device  */

	if (dev->irq == 0)
	{
		del_timer(&dev->tlnode);
	}
	/*  Wake up any processes that might be waiting for a frame  */
	/*  and let them return an error  */
	wake_up_interruptible(&dev->new_video_frame);
}

static void
interrupt_enable(struct capture_device *dev)
{
	if (dev->ints_enabled)
		interrupt_disable(dev);
	dev->ints_enabled = 1;

	/*  TODO: Enable interrupts on the device  */

	dev->tlnode.function = NULL; /* NULL indicates h/w interrupts */
	if (dev->irq == 0)
	{
		init_timer(&dev->tlnode);
		dev->tlnode.function = 
			(void(*)(unsigned long))interrupt_handler;
		dev->tlnode.data = (unsigned long)dev;
		dev->tlnode.expires = jiffies + HZ/100;
		add_timer(&dev->tlnode);
	}
	debug_msg("%s interrupts enabled\n",
		  dev->tlnode.function?"Polled":"Hardware");
}

/*
 *
 *	M E M O R Y   M A P P I N G
 *
 */

static struct stream_buffer *
mmap_stream_buffer_from_offset(struct capture_device *dev,
			       unsigned long offset)
{
	int	i;
	for (i = 0; i < MAX_CAPTURE_BUFFERS; ++i)
		if (offset == dev->stream_buf[i].vidbuf.offset)
			return &dev->stream_buf[i];
	return NULL;
}

static int
mmap_request_buffers(struct capture_device *dev,
		     struct v4l2_requestbuffers *req)
{
	int	i;
	u32	buflen;
	u32	type;

	if (dev->stream_buffers_mapped)
		return 0;/* can't make requests if buffers are mapped */
	if (req->count < 1)
		req->count = 1;
	if (req->count > MAX_CAPTURE_BUFFERS)
		req->count = MAX_CAPTURE_BUFFERS;
	type = V4L2_BUF_TYPE_CAPTURE;
	dev->stream_contig_map = 0;
	if (req->type & V4L2_BUF_REQ_CONTIG)
	{
		dev->stream_contig_map = 1;
		req->type = type | V4L2_BUF_REQ_CONTIG;
		/* note: _REQ_CONTIG is only used in v4l2_requestbuffers */
	}

	/*  The buffer length needs to be a multiple of the page size  */
	buflen = (dev->clientfmt.sizeimage + PAGE_SIZE - 1)
		& ~(PAGE_SIZE - 1);

	debug_msg("Granting %d buffers\n",req->count);

	/*  Now initialize the buffer structures. Don't allocate the */
	/*  buffers until they're mapped. */
	for (i = 0; i < req->count; ++i)
	{
		dev->stream_buf[i].requested = 1;
		dev->stream_buf[i].vidbuf.index = i;
		dev->stream_buf[i].vidbuf.type = type;
		dev->stream_buf[i].vidbuf.offset = 4*i;/* anything unique */
		dev->stream_buf[i].vidbuf.length = buflen;
		dev->stream_buf[i].vidbuf.bytesused = 0;
		dev->stream_buf[i].vidbuf.timestamp.tv_sec = 0;
		dev->stream_buf[i].vidbuf.timestamp.tv_usec = 0;
		dev->stream_buf[i].vidbuf.flags = 0;
	}
	for (i = req->count; i < MAX_CAPTURE_BUFFERS; ++i)
		dev->stream_buf[i].requested = 0;
	dev->stream_buffers_requested = req->count;

	return 1;
}

static void
mmap_unrequest_buffers(struct capture_device *dev)
{
	int	i;

	if (dev->stream_buffers_requested == 0 ||
	    dev->stream_buffers_mapped)
		return;
	for (i = 0; i < MAX_CAPTURE_BUFFERS; ++i)
		dev->stream_buf[i].requested = 0;
	dev->stream_buffers_requested = 0;
}

static void
mmap_vma_open(struct vm_area_struct *vma)
{
	struct capture_device *dev =
		capture_device_from_file(vma->vm_file);
	if (dev == NULL)
		return;
	//debug_msg("vma_open called\n");
	//MOD_INC_USE_COUNT;
}

static void
mmap_vma_close(struct vm_area_struct *vma)
{
	struct capture_device *dev =
		capture_device_from_file(vma->vm_file);
	struct stream_buffer *buf =
		mmap_stream_buffer_from_offset(dev, vma->vm_offset);
	int	i, n = 1;

	if (dev->stream_contig_map)
	{/*	Unmap all the buffers in one stroke  */
		n = dev->stream_buffers_mapped;
		buf = &dev->stream_buf[0];
	}
	for (i = 0; i < n; ++i)
	{
		if (dev->streaming)
		{
			info_msg("Warning- munmap() called while streaming\n");
			capture_streamoff(dev, buf->vidbuf.type);
		}
		v4l2_q_yank_node(&dev->stream_q_capture, &buf->qnode);
		v4l2_q_yank_node(&dev->stream_q_done, &buf->qnode);

		if (buf->vaddress != NULL && i == 0)
			vfree(buf->vaddress);
		buf->vaddress = NULL;
		if (buf->dma_list)
			free_page((unsigned long)buf->dma_list);
		buf->dma_list = NULL;
		buf->vidbuf.flags = 0;
		//debug_msg("Buffer %d deallocated\n",(int)vma->vm_offset/4);
		++buf;
		if (dev->stream_buffers_mapped > 0)
			--dev->stream_buffers_mapped;
	}
	//MOD_DEC_USE_COUNT;
}

static unsigned long
mmap_vma_nopage(struct vm_area_struct *vma,
		unsigned long address, int write)
{
	struct capture_device	*dev;
	struct stream_buffer	*buf;
	unsigned long		offset_into_buffer;
	unsigned long		page;
	int			n	= 1;

	dev = capture_device_from_file(vma->vm_file);
	if (dev == NULL)
		return 0;
	if (dev->stream_contig_map)
	{
		buf = &dev->stream_buf[0];
		n = dev->stream_buffers_requested;
	}	
	else
		buf = mmap_stream_buffer_from_offset(dev, vma->vm_offset);
	if (buf == NULL)
		return 0;
	offset_into_buffer = address - vma->vm_start;
	if (offset_into_buffer >= buf->vidbuf.length * n)
	{
		err_msg("Attempt to read past end of mmap() buffer\n");
		return 0;
	}
	page = v4l2_vmalloc_to_page(buf->vaddress + offset_into_buffer);
	if (page == 0)
		return 0;

	atomic_inc(&mem_map[MAP_NR(page)].count);
	return page;
}

static struct vm_operations_struct capture_vma_operations =
{
	mmap_vma_open, mmap_vma_close, NULL, NULL, NULL, NULL,
	mmap_vma_nopage,
};


/*
 *
 *	V I D E O   F O R   L I N U X   I N T E R F A C I N G
 *
 */

static int
v4l2_open(struct v4l2_device *v, int flags, void **idptr)
{
	struct capture_device *dev = (struct capture_device *)v;
	int	i, n;
	int	cap;

	for (i = 0, n = -1, cap = 0; i < MAX_OPENS; ++i)
	{
		if (!dev->open_data[i].isopen)
			n = i;/* available open_data structure */
		else if (!dev->open_data[i].noncapturing)
			cap = 1;/* another open is already capturing */
	}
	if (n == -1)/* No available open_data structures */
	{
		debug_msg("No more opens on this device\n");
		return -EBUSY;
	}

	if (flags & O_NONCAP)/*  Non-capturing open */
		dev->open_data[n].noncapturing = 1;
	else if (cap)
	{
		debug_msg("No more capturing opens on this device\n");
		return -EBUSY;
	}
	else
	{
		dev->open_data[n].noncapturing = 0;
		/*  Keep track of whether there is a capturing open  */
		++dev->capturing_opens;
		dev->perf.frames = 0;
		dev->perf.framesdropped = 0;
		dev->perf.bytesout = 0;
	}

	//MOD_INC_USE_COUNT;
	++dev->open_count;
	dev->open_data[n].isopen = 1;
	dev->open_data[n].dev = dev;
	*idptr = &dev->open_data[n];

	if (dev->open_count == 1)
	{
		if (dev->pci && dev->irq == 0)
		{
			dev->irq = dev->pci->irq;
			if (request_irq(dev->irq, interrupt_hw, SA_SHIRQ,
					dev->shortname, dev) < 0)
			{
				err_msg("Denied IRQ %d\n", dev->irq);
				dev->irq = 0;
			}
		}
		dev->ready_to_capture = 0;/* benchmark changes parameters! */
		dev->capture_completed = 0;
		dev->grabber_enabled = 0;
		v4l2_q_init(&dev->stream_q_capture);
		v4l2_q_init(&dev->stream_q_done);
	}
	debug_msg("Open succeeded\n");

	/* frame counter for test images only */
	if (!dev->open_data[n].noncapturing)
		dev->h=dev->m=dev->s=dev->f=0;
	return 0;
}

static void
v4l2_close(void *id)
{
	struct device_open *o = (struct device_open *)id;
	struct capture_device *dev = o->dev;

	if (!o->noncapturing)
	{
		--dev->capturing_opens;
		debug_msg("Close\n");
	}
	o->isopen = 0;
        --dev->open_count;
	if (dev->open_count == 0)
	{
		interrupt_disable(dev);
		capture_close(dev);
		if (dev->irq)
			free_irq(dev->irq, dev);
		dev->irq = 0;
	}
	//MOD_DEC_USE_COUNT;
}

static long
v4l2_write(void		*id, 
	     const char		*buf, 
	     unsigned long	count, 
	     int		noblock)
{
	debug_msg("Write() not handled\n");
	return -EINVAL;
}

/*  The arguments are already copied into kernel memory, so don't use
    copy_from_user() or copy_to_user() on arg.  */
static int
v4l2_ioctl(void		*id,
	   unsigned int	cmd,
	   void		*arg)
{
	struct device_open *o = (struct device_open *)id;
	struct capture_device *dev = o->dev;

//debug_msg("ioctl %d\n", _IOC_NR(cmd));
	switch(cmd)
	{
	case VIDIOC_QUERYCAP:
	{
		struct v4l2_capability *b = arg;
		strcpy(b->name, dev->v.name);
		b->type = V4L2_TYPE_CAPTURE;
		b->flags = V4L2_FLAG_READ |
			  V4L2_FLAG_STREAMING |
			  V4L2_FLAG_SELECT;
		b->inputs = dev->videc.num_inputs;
		b->outputs = 0;
		b->audios = 0;
		b->maxwidth = MAX_WIDTH;
		b->maxheight = MAX_HEIGHT;
		b->minwidth = MIN_WIDTH;
		b->minheight = MIN_HEIGHT;
		b->maxframerate = 30;
		return 0;
	}

	case VIDIOC_ENUM_CAPFMT:
	{
		struct v4l2_fmtdesc *f = arg;
		if (f->index < 0 || f->index >= NUM_CAPFMT)
			return -EINVAL;
		*f = capfmt[f->index];
		return 0;
	}

	case VIDIOC_G_FMT:
	{
		memcpy(arg, &dev->clientfmt, sizeof(dev->clientfmt));
		return 0;
	}

	case VIDIOC_S_FMT:
	{
		struct v4l2_format *fmt = arg;
		if (o->noncapturing)
		{
			debug_msg("S_FMT illegal in non-capturing open\n");
			return -EPERM;
		}
		dev->clientfmt = *fmt;
		if (!capture_new_format(dev))
			return -EINVAL;
		if (dev->streaming)
			capture_grab_frame(dev);
		*fmt = dev->clientfmt;
		return 0;
	}

	case VIDIOC_G_COMP:	return -EINVAL;
	case VIDIOC_S_COMP:	return -EINVAL;

	case VIDIOC_REQBUFS:
	{
		struct v4l2_requestbuffers *req = arg;
		if (o->noncapturing)
		{
			debug_msg("REQBUFS illegal in non-capturing open\n");
			return -EPERM;
		}
		if (dev->stream_buffers_mapped)
		{
			debug_msg("Can't request buffers if buffers are "
				  "already mapped\n");
			return -EPERM;
		}
		mmap_unrequest_buffers(dev);
		capture_begin(dev);
		if (!mmap_request_buffers(dev, req))
			return -EINVAL;
		return 0;
	}

	case VIDIOC_QUERYBUF:
	{
		struct v4l2_buffer *buf = arg;
		int	i;
		if (o->noncapturing)
		{
			debug_msg("QUERYBUF illegal in non-capturing open\n");
			return -EPERM;
		}
		i = buf->index;
		if (i < 0 || i >= MAX_CAPTURE_BUFFERS ||
		    !dev->stream_buf[i].requested ||
		    (buf->type & V4L2_BUF_TYPE_field) != 
		     (dev->stream_buf[i].vidbuf.type & V4L2_BUF_TYPE_field))
		{
			debug_msg("QUERYBUF bad parameter\n");
			return -EINVAL;
		}
		*buf = dev->stream_buf[i].vidbuf;
		return 0;
	}

	case VIDIOC_QBUF:
	{
		struct v4l2_buffer *buf = arg;
		if (o->noncapturing)
		{
			debug_msg("QBUF illegal in non-capturing open\n");
			return -EPERM;
		}
		if (!dev->stream_buffers_mapped)
		{
			debug_msg("QBUF no buffers are mapped\n");
			return -EINVAL;
		}
		if (!capture_queuebuffer(dev, buf))
			return -EINVAL;
		return 0;
	}

	case VIDIOC_DQBUF:
	{
		struct v4l2_buffer *buf = arg;
		if (o->noncapturing)
		{
			debug_msg("DQBUF illegal in non-capturing open\n");
			return -EPERM;
		}
		if (!capture_dequeuebuffer(dev, buf))
			return -EINVAL;
		return 0;
	}

	case VIDIOC_STREAMON:
	{
		__u32	type = (__u32)arg;
		if (o->noncapturing)
		{
			debug_msg("STREAMON illegal in non-capturing open\n");
			return -EPERM;
		}
		if (!capture_streamon(dev, type))
			return -EINVAL;
		return 0;
	}

	case VIDIOC_STREAMOFF:
	{
		__u32	type = (__u32)arg;
		if (o->noncapturing)
		{
			debug_msg("STREAMOFF illegal in non-capturing open\n");
			return -EPERM;
		}
		capture_streamoff(dev, type);
		return 0;
	}

	case VIDIOC_ENUM_FBUFFMT:	return -EINVAL;
	case VIDIOC_G_FBUF:		return -EINVAL;
	case VIDIOC_S_FBUF:		return -EINVAL;
	case VIDIOC_G_WIN:		return -EINVAL;
	case VIDIOC_S_WIN:		return -EINVAL;
	case VIDIOC_PREVIEW:		return -EINVAL;

	case VIDIOC_G_PERF:
	{
		memcpy(arg, &dev->perf, sizeof(dev->perf));
		return 0;
	}

	case VIDIOC_G_INPUT:
	{
		memcpy(arg, &dev->input, sizeof(dev->input));
		return 0;
	}

	case VIDIOC_S_INPUT:
	{
		int	input = (int)arg;
		if (input < 0 || input >= dev->videc.num_inputs)
		{
			debug_msg("Input out of range %d\n", input);
			return -EINVAL;
		}
		if (input != dev->input)
		{
			dev->input = input;
			set_video_input(dev, input);
		}
		return 0;
	}

	case VIDIOC_G_PARM:
	{
		memcpy(arg, &dev->capture, sizeof(dev->capture));
		return 0;
	}

	case VIDIOC_S_PARM:
	{
		struct v4l2_captureparm *vp = arg;
		if (vp->capturemode & ~dev->capture.capability)
		{
			debug_msg("PARM unsupported capture capability\n");
			return -EINVAL;
		}
		if ((dev->capture.capability & V4L2_CAP_TIMEPERFRAME) &&
		    vp->timeperframe < 10000)
		{
			debug_msg("PARM time per frame out of range %ld\n",
				  vp->timeperframe);
			return -EINVAL;
		}
		if (vp->capturemode != dev->capture.capturemode &&
		    !o->noncapturing && dev->streaming)
			return -EINVAL;

		if (o->noncapturing)
			return 0;
		if (vp->capturemode != dev->capture.capturemode)
		{
			dev->capture.capturemode = vp->capturemode;
			capture_new_format(dev);
		}
		if ((vp->capturemode & V4L2_CAP_TIMEPERFRAME) &&
		    vp->timeperframe >= dev->videc.frame_period)
			dev->capture.timeperframe = vp->timeperframe;
		else
			dev->capture.timeperframe = dev->videc.frame_period;
		return 0;
	}

	case VIDIOC_G_STD:
	{
		struct v4l2_standard *std = arg;
		v4l2_video_std_construct(std, dev->videc.standard, 0);
		return 0;
	}

	case VIDIOC_S_STD:
	{
		struct v4l2_standard	*std = arg;
		int			id;
		if ((o->noncapturing && dev->capturing_opens) ||
		    dev->stream_buffers_mapped)
			return -EPERM;
		id = v4l2_video_std_confirm(std);
		if (!((1 << id) & dev->videc.standards))
		{
			debug_msg("Bad standard: %u\n", (unsigned)id);
			return -EINVAL;
		}
		dev->videc.set_standard(dev, id);
		return 0;
	}

	case VIDIOC_ENUMSTD:
	{
		struct v4l2_enumstd *estd = arg;
		__u32	b, i;
		if (estd->index < 0 || estd->index > 30)
			return -EINVAL;
		for (b = 1, i = 0; b < 32; ++b)
		{
			if (((1 << b) & dev->videc.standards) == 0)
				continue;
			if (i == estd->index)
			{
				v4l2_video_std_construct(&estd->std, b, 0);
				estd->inputs = (__u32)-1; /* all inputs */
				estd->outputs = 0;
				return 0;
			}
			++i;
		}
		return -EINVAL;
	}

	case VIDIOC_ENUMINPUT:
	{
		struct v4l2_input *vi = arg;
		if (vi->index < 0 || vi->index >= dev->videc.num_inputs)
			return -EINVAL;
		*vi = dev->source[vi->index].input;
		return 0;
	}

	case VIDIOC_QUERYCTRL:
	{
		struct v4l2_queryctrl	*qc = arg;
		int			i;
		i = find_vctrl(qc->id);
		if (i < 0)
		{
			return i;
		}
		/*  V4L2 filled in category and catname, preserve them */
		capture_control[i].category = qc->category;
		memcpy(capture_control[i].catname, qc->catname, 
		       sizeof(qc->catname));
		*qc = capture_control[i];
		return 0;
	}

	case VIDIOC_QUERYMENU:
	{
		struct v4l2_querymenu	*qm = arg;

		return vctrl_querymenu(qm);
	}

	case VIDIOC_G_CTRL:
	{
		struct v4l2_control	*vc = arg;
		int			i;
		i = find_vctrl(vc->id);
		if (i < 0)
			return i;
		vc->value = dev->source[dev->input].control[i];
		return 0;
	}

	case VIDIOC_S_CTRL:
	{
		struct v4l2_control	*vc = arg;
		int			i;
		i = find_vctrl(vc->id);
		if (i < 0)
			return i;
		dev->source[dev->input].control[i] = vc->value;
		device_tone_controls(dev);
		return 0;
	}

	case VIDIOC_G_TUNER:	return -EINVAL;
	case VIDIOC_S_TUNER:	return -EINVAL;
	case VIDIOC_G_FREQ:	return -EINVAL;
	case VIDIOC_S_FREQ:	return -EINVAL;

	case VIDIOC_G_AUDIO:	return -EINVAL;
	case VIDIOC_S_AUDIO:	return -EINVAL;

	default:
		return -ENOIOCTLCMD;
	}
	return 0;
}

static int
v4l2_mmap(void			*id,
	  struct vm_area_struct *vma)
{
	struct device_open	*o   = (struct device_open *)id;
	struct capture_device	*dev = o->dev;
	struct stream_buffer	*buf;
	int			i, n = 1;

	if (o->noncapturing)
	{
		debug_msg("mmap() called on non-capturing open\n");
		return -ENODEV;
	}
	buf = mmap_stream_buffer_from_offset(dev, vma->vm_offset);
	if (dev->stream_contig_map)
	{/*	N buffers in one contiguous map  */
		buf = &dev->stream_buf[0];
		n = dev->stream_buffers_requested;
	}
	if (buf == NULL)
	{
		debug_msg("mmap() Invalid offset parameter\n");
		return -EINVAL;/* no such buffer */
	}
	if (buf->vidbuf.length * n != vma->vm_end - vma->vm_start)
	{
		debug_msg("mmap() Wrong length parameter\n");
		return -EINVAL;/* wrong length */
	}
	for (i = 0; i < n; ++i)
	{
		if (!buf->requested)
		{
			debug_msg("mmap() Buffer is not available for"
				  " mapping\n");
			return -EINVAL;/* not requested */
		}
		if (buf->vidbuf.flags & V4L2_BUF_FLAG_MAPPED)
		{
			debug_msg("mmap() Buffer is already mapped\n");
			return -EINVAL;/* already mapped */
		}

		if (buf->vaddress != NULL)
			vfree(buf->vaddress);
		if (i == 0)
			buf->vaddress = vmalloc(buf->vidbuf.length * n);
		else
			buf->vaddress = buf[-1].vaddress + buf->vidbuf.length;
		if (buf->vaddress == NULL)
		{
			err_msg("Could not allocate mmap() buffer\n");
			return -ENODEV;
		}
#if 0		/*  TODO: build scatter list for buffer if using DMA  */
		if ((using DMA) &&
		    !bm_build_scatter_list(dev, buf->vaddress, &buf->dma_list))
			return -ENODEV;
#endif
		buf->vidbuf.flags |= V4L2_BUF_FLAG_MAPPED;
		++dev->stream_buffers_mapped;
		++buf;
	}

	vma->vm_ops = &capture_vma_operations;
	if (vma->vm_ops->open)
		vma->vm_ops->open(vma);
	/*  Note: vma->vm_file will be set up by V4L2  */

	return 0;
}

static int
v4l2_poll(void	*id,
	    struct file	*file,
	    poll_table	*table)
{
	struct device_open *o = (struct device_open *)id;
	struct capture_device *dev = o->dev;

	if (o->noncapturing)
	{
		debug_msg("poll() illegal in non-capturing open\n");
		return POLLERR;
	}

	if (dev->streaming)
	{
		void	*node;
		node = v4l2_q_peek_head(&dev->stream_q_done);
		if (node != NULL)
			return (POLLIN | POLLRDNORM);/* data is ready now */
		node = v4l2_q_peek_head(&dev->stream_q_capture);
		if (node == NULL)
			return POLLERR;  /* no frames queued */
		poll_wait(file, &dev->new_video_frame, table);
		return 0;
	}

	/*  Capture is through read() call */

	if (dev->capture_completed)/* data is ready now */
		return (POLLIN | POLLRDNORM);
	capture_grab_frame(dev);/* does nothing if capture is in progress */
	if (!dev->ready_to_capture)/* Can't grab frames! */
		return POLLERR;
	poll_wait(file, &dev->new_video_frame, table);
	return 0;
}

static long
v4l2_read(void		*id,
	    char		*buf,
	    unsigned long	count,
	    int			noblock)
{
	struct device_open *o = (struct device_open *)id;
	struct capture_device *dev = o->dev;
	long	len = 0;
	long	my_timeout;

	if (o->noncapturing)
	{
		debug_msg("read() illegal in non-capturing open\n");
		return -EPERM;
	}
	if (dev->streaming)
	{
		debug_msg("Can't read() when streaming is on\n");
		return -EPERM;
	}
	capture_grab_frame(dev);/* does nothing if capture is in progress */
	if (!dev->ready_to_capture)
	{
		debug_msg("Can't grab frames!\n");
		return 0;
	}

	my_timeout = HZ / 5;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,1,127)
	current->timeout = jiffies + my_timeout;
#endif
	while (len == 0)
	{
		if (noblock)
		{
			if (!dev->capture_completed)
				return -EAGAIN;
		}
		else
		{
			/* watch out for race condition going to sleep! */
			cli();
			if (!dev->capture_completed)
			{
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,1,127)
				interruptible_sleep_on(&dev->new_video_frame);
#else
				my_timeout = interruptible_sleep_on_timeout(
					&dev->new_video_frame, my_timeout);
#endif
			}
			sti();
		}
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,1,127)
		if (current->timeout <= jiffies)
#else
		if (my_timeout == 0)
#endif
		{
			debug_msg("Timeout on read\n");
			break;
		}
		len = capture_read(dev, buf, count);
	}
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,1,127)
	current->timeout = 0;
#endif
	//debug_msg("read %d\n", (int)len);
	return len;
}

/*
 *	Remaining initialization of video decoder etc. This is only
 *	done when the device is successfully identified and registered.
 */
static int
v4l2_init_done(struct v4l2_device *v)
{
	struct capture_device *dev = (struct capture_device *)v;
	int	i;

	/*  Initialize video input array	*/
	for (i = 0; i < VSOURCE_COUNT; ++i)
	{
		dev->source[i].input.index = i;
		dev->source[i].input.type = V4L2_INPUT_TYPE_CAMERA;
		dev->source[i].input.capability = 0;
		/*  Initialize video control properties	*/
		dev->source[i].control[VCTRL_BRIGHTNESS] =
			capture_control[VCTRL_BRIGHTNESS].default_value;
		dev->source[i].control[VCTRL_CONTRAST] =
			capture_control[VCTRL_CONTRAST].default_value;
		dev->source[i].control[VCTRL_SATURATION] =
			capture_control[VCTRL_SATURATION].default_value;
/* 		dev->source[i].control[VCTRL_HUE] = */
/* 			capture_control[VCTRL_HUE].default_value; */
	}
	strcpy(dev->source[VSOURCE_COMP].input.name, "Composite");
	strcpy(dev->source[VSOURCE_SVIDEO].input.name, "S-Video");
	/*strcpy(dev->source[VSOURCE_TUNER].input.name, "Tuner");*/
	/*dev->source[VSOURCE_TUNER].input.type = V4L2_INPUT_TYPE_TUNER;*/

	/*  Initialize the video decoder hardware	*/
	dev->videc.initialize(dev);

	/*  BUG: get defaults from user somehow...  */
	dev->videc.set_standard(dev, V4L2_STD_NTSC);
	dev->videc.set_vcrmode(dev, 0);
	set_video_input(dev, VSOURCE_COMP);

	/*  Capture parameters  */
	dev->capture.capability = V4L2_CAP_TIMEPERFRAME;
	dev->capture.capturemode = 0;
	dev->capture.extendedmode = 0;
	dev->capture.timeperframe = dev->videc.frame_period;

	/*  Default capture dimensions	*/
	dev->clientfmt.width = 160;
	dev->clientfmt.height = 120;
	dev->clientfmt.depth = 16;
	dev->clientfmt.pixelformat = V4L2_PIX_FMT_RGB565;
	dev->clientfmt.flags = 0;
	dev->clientfmt.bytesperline = 0;
	dev->clientfmt.sizeimage = 0;
	capture_new_format(dev);

	init_waitqueue(&dev->new_video_frame);
	return 0;
}

/*  =====================================================================
 *	The functions below this point are only called during loading
 *	and unloading of the driver.
 */


/*
 *	D E V I C E   I N I A L I Z A T I O N   R O U T I N E S
 *
 *	These routines locate and enable the hardware, and initialize
 *	the device structure. 
 */




#if 0
/*	Variables for assigning resources via the command line
 */
/*  ISA non-PnP IO base overrides */
static int	isa0_iobase	= 0;
static int	isa1_iobase	= 0;
#ifdef MODULE_PARM
MODULE_PARM(isa0_iobase, "i");
MODULE_PARM(isa1_iobase, "i");
#endif

/*  ISA PnP IO base override */
static int	pnp0_iobase	= 0;
#ifdef MODULE_PARM
MODULE_PARM(pnp0_iobase, "i");
#endif
#endif

/*  Initialize v4l2_device fields	*/
static int
init_device_fields(struct capture_device *dev)
{
	int num = dev - capture;

	sprintf(dev->v.name, "Example Capture Driver (%d)", num);
	dev->v.type = V4L2_TYPE_CAPTURE;
	dev->v.minor = unit_video[num];

	dev->v.open = v4l2_open;
	dev->v.close = v4l2_close;
	dev->v.read = v4l2_read;
	dev->v.write = v4l2_write;
	dev->v.ioctl = v4l2_ioctl;
	dev->v.mmap = v4l2_mmap;
	dev->v.poll = v4l2_poll;
	dev->v.initialize = v4l2_init_done;
	dev->v.priv = NULL;
	return 1;/* OK */
}

static int
config_a_device(struct capture_device *dev)
{
	sprintf(dev->shortname, "capture%d", dev - capture);

	/*  TODO: Search for an unconfigured device, configure the  */
	/*        I/O port  */
	/* if (!(found another device)) */
	/*	return 0; */

	device_initialize(dev);
	if (!init_device_fields(dev))
		return 0;

	if (!find_decoder(dev))
	{
		err_msg("Bad or unrecognized video decoder\n");
		return 0;/* failed */
	}
	return 1;
}

static void
unconfig_a_device(struct capture_device *dev)
{
	interrupt_disable(dev);
	capture_close(dev);

	/*  TODO: Unconfigure the device, free the I/O port, etc.  */

	if (dev->is_registered)
	{
		v4l2_unregister_device((struct v4l2_device *)dev);
		info_msg("Removed device %s\n", dev->shortname);
	}
	memset(dev, 0, sizeof(capture[0]));
}

/*
 *	M O D U L E   I N I T   A N D   C L E A N U P
 */

int
init_module(void)
{
	int	i;

	for (i = 0; i < NBOARDS; ++i)
	{
		memset(&capture[i], 0, sizeof(capture[0]));
		if (!config_a_device(&capture[i]))
		{
			break;
		}
		if (v4l2_register_device(
			(struct v4l2_device *)&capture[i]) != 0)
		{
			err_msg("Couldn't register the driver.\n");
			unconfig_a_device(&capture[i]);
			return 0;
		}
		capture[i].is_registered = 1;
	}
	if (i == 0)
	{
		err_msg("No devices found.\n");
		return -ENODEV;/* cleanup will not be called */
	}
	return 0;
}

void
cleanup_module(void)
{
	int	i;

	for (i = 0; i < NBOARDS; ++i)
		unconfig_a_device(&capture[i]);
}

/*===================================================================
  ===================================================================

 *  Since this example has no capture hardware, this code will make
    synthetic images so there is some data to look at
*/

void
toneline(struct capture_device *dev,
	 __u8 *p, int n)
{
	int	*ctrl;
	__u8	*p1;
	int	b, c, s;
	int	i, t;

	ctrl = dev->source[dev->input].control;
	b = ctrl[VCTRL_BRIGHTNESS];
	c = ctrl[VCTRL_CONTRAST];
	s = ctrl[VCTRL_SATURATION];

	/*  Brightness and contrast  */
	for (i = 0, p1 = p; i < n; ++i, p1 += 2)
	{
		t = (((*p1 - 16) * c + 16) / 32) + 16 + b - 32;
		if (t < 16) t = 16;
		if (t > 235) t = 235;
		*p1 = t;
	}
	if (n == 1)
		return;
	/*  Saturation  */
	for (i = 0, p1 = p + 1; i < n; ++i, p1 += 2)
	{
		t = (((int)*p1 - 128) * s + 16) / 32 + 128;
		if (t < 16) t = 16;
		if (t > 239) t = 239;
		*p1 = t;
	}
}

static void
vline(__u8 *p, int stride, int k, int w, u8 m)
{
	int i;
	p += stride;
	for (i = 0; i < k; ++i)
	{
		p[0] = m;
		p[2] = m;
		if (w >= 3) p[4] = m;
		p += stride;
	}
}
static void
hline(__u8 *p, int stride, int k, int w, u8 m)
{
	int i;
	p += 2;
	for (i = 0; i < k; ++i)
	{
		p[0] = m;
		p[stride] = m;
		if (w >= 3) p[2*stride] = m;
		p += 2;
	}
}
static void
dline(__u8 *p, int stride, int a, int k, int w, u8 m)
{
	switch (a)
	{
	case 0: hline(p, stride, k, w, m); break;
	case 1: hline(p + k * stride, stride, k, w,m ); break;
	case 2: hline(p + 2 * k * stride, stride, k, w, m); break;
	case 3: vline(p, stride, k, w, m); break;
	case 4: vline(p + 2 * k, stride, k, w, m); break;
	case 5: vline(p + k * stride, stride, k, w, m); break;
	case 6: vline(p + k * stride + 2 * k, stride, k, w, m); break;
	}
}
static void
digit(__u8 *capture_buffer, int stride, char c, int x, int y,
      int k, int w, u8 m)
{
	__u8 *p = capture_buffer + 2 * x + y * stride;
	int	i, t = 0;
	switch (c)
	{
	case '0': t = 0x7d; break;
	case '1': t = 0x50; break;
	case '2': t = 0x37; break;
	case '3': t = 0x57; break;
	case '4': t = 0x5a; break;
	case '5': t = 0x4f; break;
	case '6': t = 0x6f; break;
	case '7': t = 0x51; break;
	case '8': t = 0x7f; break;
	case '9': t = 0x5b; break;
	case ':': hline(p + 2 * (k/2) + (k/2) * stride, stride, w, w, m);
		  hline(p + 2 * (k/2) + 3 * (k/2) * stride, stride, w, w, m);
		  return;
	}
	for (i = 0; t && i < 7; ++i)
		if (t & (1 << i))
			dline(p, stride, i, k, w, m);
}
static void
synthetic_image(struct capture_device *dev, __u8 *capture_buffer)
{
	char	st[16];
	int	width	= dev->clientfmt.width;
	int	height	= dev->clientfmt.height;
	int	stride	= 2 * width;
	int	i, n, b = 0;
	int	k, dw, lm, lw = 3;
	u32	*q;
	u8	*p	= capture_buffer;
	u8	m	= 0xff;

	++dev->f;
	if (dev->videc.frame_period == 400000)
	{
		if (dev->f == 25) { dev->f = 0; ++dev->s; }
	}
	else
		if (dev->f == 30) { dev->f = 0; ++dev->s; }
	if (dev->s == 60) { dev->s = 0; ++dev->m; }
	if (dev->m == 60) { dev->m = 0; ++dev->h; }
	if (dev->h == 1000) { dev->h = 0; }
	n = sprintf(st, "%03d:%02d:%02d:%02d", dev->h,dev->m,dev->s,dev->f);

	dw = width / (n + 1);
	if (dw > (height >> 3))
		dw = height >> 3;
	if (dw < 10)
	{
		dw = 10;
		b = n - (width / dw - 1);
	}
	k = dw - 7;
	if (k < 8) lw = 2;
	if (height < 4 * dw || width < 3 * dw)
		k = 1;

	for (i = 0, q = (u32 *)p; i < width >> 1; ++i)
	{/*	ITU-R BT.801 100/75 color bars  */
		if      (i < ((1 * width) >> 4)) *q++ = 0x80EB80EB;
		else if (i < ((2 * width) >> 4)) *q++ = 0x8EA22CA2;
		else if (i < ((3 * width) >> 4)) *q++ = 0x2C839C83;
		else if (i < ((4 * width) >> 4)) *q++ = 0x3A704870;
		else if (i < ((5 * width) >> 4)) *q++ = 0xC654B854;
		else if (i < ((6 * width) >> 4)) *q++ = 0xD4416441;
		else if (i < ((7 * width) >> 4)) *q++ = 0x7223D423;
		else				 *q++ = 0x80108010;
	}
	toneline(dev, p, width);
	for (i = 1; i < height - 4 * k; ++i, q += stride >> 2)
		memcpy(q, p, stride);
	for (i = 0, p = (u8 *)q; i < width >> 1; ++i)
	{
		*q++ = 0x80808080;
	}
	toneline(dev, p, width);
	for (i = height - 4 * k + 1; i < height; ++i, q += stride >> 2)
		memcpy(q, p, stride);

	if (k < 3)
		return;
	lm = (width - dw * (n - b)) / 2;
	toneline(dev, &m, 1);
	for (i = b; i < n; ++i)
		digit(capture_buffer, stride, st[i], lm + (i - b) * dw, 
		      height - 3 * k, k, lw, m);
}

